<?php

namespace App\Http\Controllers\Vendor;

use Exception;
use Illuminate\Http\Request;
use App\Constants\GlobalConst;
use App\Models\Admin\SetupKyc;
use Illuminate\Support\Facades\DB;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Auth;
use App\Traits\ControlDynamicInputFields;
use Illuminate\Support\Facades\Validator;
use App\Models\Vendor\VendorAuthorization;
use App\Providers\Admin\BasicSettingsProvider;
use Illuminate\Validation\ValidationException;
use App\Http\Helpers\Api\Helpers as ApiResponse;
use App\Models\Vendor\Vendor;
use Carbon\Carbon;
use App\Notifications\User\Auth\SendAuthorizationCode;

class AuthorizationController extends Controller
{
    use ControlDynamicInputFields;
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function showMailFrom($token)
    {
        $page_title = setPageTitle(__("Mail Authorization"));
        $email = VendorAuthorization::where('token', $token)
            ->join('vendors', 'vendor_authorizations.vendor_id', '=', 'vendors.id')
            ->value('vendors.email');
        return view('vendor.auth.authorize.verify-mail', compact("page_title", "token", "email"));
    }

    /**
     * Verify authorizaation code.
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function mailVerify(Request $request, $token)
    {
        $request->merge(['token' => $token]);
        $request->validate([
            'token'     => "required|string|exists:vendor_authorizations,token",
            'code'      => "required",
        ]);

        $code = implode($request->code);
        $otp_exp_sec = BasicSettingsProvider::get()->otp_exp_seconds ?? GlobalConst::DEFAULT_TOKEN_EXP_SEC;
        $auth_column = VendorAuthorization::where("token", $request->token)->where("code", $code)->first();
        if (!$auth_column) {
            return redirect()->back()->with(['error' => ['Invalid otp code']]);
        }
        if ($auth_column->created_at->addSeconds($otp_exp_sec) < now()) {
            $this->authLogout($request);
            return redirect()->route('vendors.login')->with(['error' => [__('Session expired. Please try again')]]);
        }

        try {
            $auth_column->vendor->update([
                'email_verified'    => true,
            ]);
            $auth_column->delete();
        } catch (Exception $e) {
            $this->authLogout($request);
            return redirect()->route('vendors.login')->with(['error' => [__('Something went wrong! Please try again')]]);
        }

        return redirect()->intended(route("vendors.dashboard"))->with(['success' => [__('Account successfully verified')]]);
    }



    public function authLogout(Request $request)
    {
        auth()->guard("vendor")->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
    }


    public function mailResend($token)
    {
        $vendor_authorize = VendorAuthorization::where("token", $token)->first();
        if (!$vendor_authorize) return back()->with(['error' => ['Request token is invalid']]);
        if (Carbon::now() <= $vendor_authorize->created_at->addMinutes(GlobalConst::USER_PASS_RESEND_TIME_MINUTE)) {
            throw ValidationException::withMessages([
                'code'      => 'You can resend verification code after ' . Carbon::now()->diffInSeconds($vendor_authorize->created_at->addMinutes(GlobalConst::USER_PASS_RESEND_TIME_MINUTE)) . ' seconds',
            ]);
        }
        $resend_code = generate_random_code();
        try {
            $vendor_authorize->update([
                'code'          => $resend_code,
                'created_at'    => now(),
            ]);
            $data = $vendor_authorize->toArray();
            try {
                $vendor_authorize->vendor->notify(new SendAuthorizationCode((object) $data));
            } catch (Exception $e) {
            }
        } catch (Exception $e) {
            throw ValidationException::withMessages([
                'code'      => "Something went wrong! Please try again.",
            ]);
        }

        return redirect()->route('vendors.authorize.mail', $token)->with(['success' => ['Mail Resend Success!']]);
    }



    //  2fa Verify

    public function showGoogle2FAForm()
    {
        $page_title =  __("Authorize Google Two Factor");
        return view('vendor.auth.authorize.verify-g2fa', compact('page_title'));
    }

    public function google2FASubmit(Request $request)
    {
        $request->validate([
            'code*'    => "required|numeric",
        ]);
        $code = implode($request->code);
        $vendor = auth()->user();
        if (!$vendor->two_factor_secret) {
            return back()->with(['warning' => [__('Your secret key not stored properly. Please contact with system administrator')]]);
        }

        if (google_2fa_verify($vendor->two_factor_secret, $code)) {
            $vendor->update([
                'two_factor_verified'   => true,
            ]);
            return redirect()->intended(route('vendors.dashboard'));
        }
        return back()->with(['warning' => [__('Failed to login. Please try again')]]);
    }

    public function showKycFrom()
    {
        $username = auth()->user()->username;

        $users = Vendor::with('kyc')->where("username", $username)->first();
        $page_title = setPageTitle(__("KYC Verification"));
        $vendor_kyc = SetupKyc::VendorKyc()->first();


        if (!$vendor_kyc) return back();
        $kyc_data = $vendor_kyc->fields;
        $kyc_fields = [];
        if ($kyc_data) {
            $kyc_fields = array_reverse($kyc_data);
        }
        return view('vendor.sections.verify-kyc', compact(
            "page_title",
            "kyc_fields",
            "users"
        ));
    }

    public function kycSubmit(Request $request)
    {
        $vendor = auth()->user();
        if ($vendor->kyc_verified == GlobalConst::VERIFIED) return back()->with(['success' => [__('You are already KYC Verified User')]]);

        $user_kyc_fields = SetupKyc::VendorKyc()->first()->fields ?? [];
        $validation_rules = $this->generateValidationRules($user_kyc_fields);

        $validated = Validator::make($request->all(), $validation_rules)->validate();
        $get_values = $this->placeValueWithFields($user_kyc_fields, $validated);

        $create = [
            'vendor_id'      => auth()->user()->id,
            'data'          => json_encode($get_values),
            'created_at'    => now(),
        ];

        DB::beginTransaction();
        try {
            DB::table('vendor_kyc_data')->updateOrInsert(["vendor_id" => $vendor->id], $create);
            $vendor->update([
                'kyc_verified'  => GlobalConst::PENDING,
            ]);
            DB::commit();
        } catch (Exception $e) {
            DB::rollBack();
            $vendor->update([
                'kyc_verified'  => GlobalConst::DEFAULT,
            ]);
            $this->generatedFieldsFilesDelete($get_values);
            return back()->with(['error' => [__('Something went wrong! Please try again')]]);
        }

        return redirect()->route('vendors.authorize.kyc')->with(['success' => [__('KYC information successfully submitted')]]);
    }


    public function logout()
    {

        Auth::user()->token()->revoke();
        $message = ['success' => [__('Logout Successful')]];
        return ApiResponse::success($message);
    }
}
