<?php

namespace App\Http\Controllers\Vendor\Auth;

use App\Http\Controllers\Controller;
use App\Models\Vendor\Vendor;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Auth\Events\Registered;
use App\Traits\Vendor\RegisteredVendors;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use App\Providers\Admin\BasicSettingsProvider;
use Illuminate\Foundation\Auth\RegistersUsers;
use Illuminate\Http\Request;
use App\Traits\Vendor\LoggedInVendors;
use Exception;
use App\Models\Vendor\VendorWallet;
use App\Models\Admin\Currency;
use Illuminate\Support\Str;

class RegisterController extends Controller
{
    /*
    |--------------------------------------------------------------------------
    | Register Controller
    |--------------------------------------------------------------------------
    |
    | This controller handles the registration of new users as well as their
    | validation and creation. By default this controller uses a trait to
    | provide this functionality without requiring any additional code.
    |
    */

    use RegistersUsers, RegisteredVendors,LoggedInVendors;

    protected $basic_settings;

    public function __construct()
    {
        $this->basic_settings = BasicSettingsProvider::get();
    }

    /**
     * Show the application registration form.
     *
     * @return \Illuminate\View\View
     */
    public function showRegistrationForm() {
        $client_ip = request()->ip() ?? false;
        $vendor_country = geoip()->getLocation($client_ip)['country'] ?? "";

        $page_title = setPageTitle("Vendor Registration");
        return view('vendor.auth.register',compact(
            'page_title',
            'vendor_country',
        ));
    }


    /**
     * Handle a registration request for the application.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\RedirectResponse|\Illuminate\Http\JsonResponse
     */
    public function register(Request $request)
    {
        $validated                          = $this->validator($request->all())->validate();
        $basic_settings                     = $this->basic_settings;
        $validated['email_verified']        = ($basic_settings->vendor_email_verification == true) ? false : true;
        $validated['sms_verified']          = ($basic_settings->vendor_sms_verification == true) ? false : true;
        $validated['kyc_verified']          = ($basic_settings->vendor_kyc_verification == true) ? false : true;
        $validated['password']              = Hash::make($validated['password']);
        $validated['username']              = make_username(Str::slug($validated['firstname']),Str::slug($validated['lastname']));

        event(new Registered($user = $this->create($validated)));
        $this->guard()->login($user);

            return $this->registered($request, $user);
    }
    /**
     * Get a validator for an incoming registration request.
     *
     * @param  array  $data
     * @return \Illuminate\Contracts\Validation\Validator
     */
    public function validator(array $data) {

        $basic_settings = $this->basic_settings;
        $passowrd_rule = "required|string|min:6";
        if($basic_settings->vendor_secure_password) {
            $passowrd_rule = ["required",Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised()];
        }
        $agree_rule = $basic_settings->vendor_agree_policy ? 'required|in:on' : 'in:on';

        return Validator::make($data,[
            'firstname'     => 'required|string|max:60',
            'lastname'      => 'required|string|max:60',
            'email'         => 'required|string|email|max:150|unique:vendors,email',
            'password'      => $passowrd_rule,
            'agree'         => $agree_rule,
        ]);
    }


    /**
     * Create a new user instance after a valid registration.
     *
     * @param  array  $data
     * @return \App\Models\User
     */
    protected function create(array $data)
    {
        return Vendor::create($data);
    }


    /**
     * Get the guard to be used during registration.
     *
     * @return \Illuminate\Contracts\Auth\StatefulGuard
     */
    protected function guard()
    {
        return Auth::guard('vendor');
    }
    protected function createUserWallets($user) {
        $currencies = Currency::active()->roleHasOne()->pluck("id")->toArray();
        $wallets = [];
        foreach($currencies as $currency_id) {
            $wallets[] = [
                'vendor_id'     => $user->id,
                'currency_id'   => $currency_id,
                'balance'       => 0,
                'status'        => true,
                'created_at'    => now(),
            ];
        }

        try{
            VendorWallet::insert($wallets);
        }catch(Exception $e) {
            // handle error
            throw new Exception("Failed to create wallet! Please try again");
        }
    }

    /**
     * The user has been registered.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  mixed  $user
     * @return mixed
     */
    protected function registered(Request $request, $vendor)
    {

        try{
            $this->createUserWallets($vendor);
            return redirect()->intended(route('vendors.dashboard'));
        }catch(Exception $e){

        }

    }
}
