<?php

namespace App\Http\Controllers\User;

// use PhpOffice\PhpSpreadsheet\Writer\Pdf;

use PDF;
use App\Http\Controllers\Controller;
use App\Http\Helpers\Response;
use Illuminate\Http\Request;
use App\Models\ParlourBooking;
use App\Providers\Admin\BasicSettingsProvider;
use Illuminate\Support\Facades\Validator;


class MyBookingController extends Controller
{
    /**
     * Method for view the users history page
     */
    public function index(BasicSettingsProvider  $basic_settings)
    {

        $page_title             = "My Bookings";
        $transactions           = ParlourBooking::where('user_id', auth()->id())->with(['parlour', 'schedule', 'payment_gateway', 'user'])
            ->orderBYDESC('id')
            ->paginate(10);

        return view('user.sections.my-booking.index', compact(
            'page_title',
            'transactions',
        ));
    }
    /**
     * Method for showing the history details
     * @param $slug
     * @param \Illuminate\Http\Request $request
     */
    public function details(Request $request, $slug)
    {
        $page_title     = "Booking Details";
        $user_id        = auth()->user()->id;
        $data           = ParlourBooking::where('user_id', $user_id)->with(['parlour', 'schedule', 'payment_gateway'])->where('slug', $slug)->first();
        return view('user.sections.my-booking.details', compact(
            'page_title',
            'data',
        ));
    }
    /**
     * Method for search booking log
     */
    public function search(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'text'  => 'required|string',
        ]);
        if ($validator->fails()) {
            return Response::error($validator->errors()->all(), [], 400);
        }

        $validated = $validator->validate();

        $transactions    = ParlourBooking::auth()->with(['parlour', 'schedule', 'payment_gateway'])->search($validated['text'])->get();


        return view('user.components.data-table.parlour-booking-table', compact('transactions'));
    }

    public function bookingReview($id, Request $request)
    {


        $request->validate([
            'rating' => 'required|integer|between:1,5',
            'comment' => 'nullable|string|max:500',
        ]);

        $booking = ParlourBooking::where('user_id', auth()->id())
            ->where('id', $id)
            ->where('status', global_const()::STATUSSUCCESS)
            ->firstOrFail();



        if ($booking->review) {
            return back()->with(['error' => ['You have already submitted a review for this booking.']]);
        }

        $booking->update([
            'review' => $request->rating,
            'comment' => $request->comment,
        ]);

        return redirect()->route('user.my.booking.index')->with(['success' => ['Thank you for your review!']]);
    }

    /**
     * Download booking details as PDF
     */
    public function downloadPdf($slug)
    {
        $user_id = auth()->user()->id;
        $data = ParlourBooking::where('user_id', $user_id)
            ->with(['parlour', 'schedule', 'payment_gateway'])
            ->where('slug', $slug)
            ->firstOrFail();

        $page_title = "Booking Details";

        // Load the view with data
        $pdf = PDF::loadView('user.sections.my-booking.pdf-details', compact('page_title', 'data'));

        // Download the PDF with a filename
        return $pdf->download('booking-details-' . $data->trx_id . '.pdf');
    }
}
