<?php

namespace App\Http\Controllers\Frontend;

use App\Constants\GlobalConst;
use Exception;
use App\Models\Admin\Area;
use Illuminate\Http\Request;
use App\Models\Admin\Language;
use App\Models\Admin\UsefulLink;
use App\Models\Frontend\Subscribe;
use App\Http\Controllers\Controller;
use App\Models\Admin\ParlourList;
use App\Models\Admin\Blog;
use App\Models\Admin\BlogCategory;
use App\Models\Admin\SetupPage;
use App\Models\Frontend\ContactRequest;
use Illuminate\Support\Facades\Session;
use Illuminate\Support\Facades\Validator;
use App\Providers\Admin\BasicSettingsProvider;

class IndexController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function index(BasicSettingsProvider $basic_settings)
    {
        $areas                  = Area::where('status', true)->get();
        $parlour_lists          = ParlourList::where('status', true)->with('favorites')->where('active_status', GlobalConst::STATUSUNFREEZE)->latest()->take(6)->get();
        return view('frontend.index', compact('areas', 'parlour_lists'));
    }
    /**
     * Method for view the find parlour page
     * @return view
     */
    public function findParlour(Request $request)
    {
        $page_title             = "Find Salon";
        $areas                  = Area::where('status', true)->get();
        $parlour_lists          = ParlourList::where('status', true)->where('active_status', GlobalConst::STATUSUNFREEZE)->paginate(6);

        $message                = Session::get('message');
        $validator = Validator::make($request->all(), [
            'area'          => 'nullable',
            'name'          => 'nullable',
        ]);
        if ($validator->fails()) {
            return back()->with(['error' => ['Something went wrong! Please try again.']]);
        }
        if ($request->area && $request->name) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->where('name', 'like', '%' . $request->name . '%')->get();
        } else if ($request->area) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->get();
        } else if ($request->name) {
            $parlour_lists    = ParlourList::where('name', 'like', '%' . $request->name . '%')->get();
        }
        $areaString     = $request->area;
        $nameString     = $request->name;
        return view('frontend.pages.find-parlour', compact(
            'page_title',
            'areas',
            'areaString',
            'nameString',
            'parlour_lists',
            'message'
        ));
    }
    /**
     * Method for view the about page
     * @return view
     */
    public function vendor()
    {
        $page_title           = "Vendor";
        $page_section       = SetupPage::where('slug', 'vendor')->with(['sections' => function ($q) {
                $q->where('status', true);
            }])->first();
          
        return view('frontend.pages.vendor', compact(
            'page_title',
            'page_section'
        ));
    }

    /**
     * Method for view the about page
     * @return view
     */
    public function about()
    {
        $page_title             = "About";
        $page_section       = SetupPage::where('slug', 'about-us')->with(['sections' => function ($q) {
                $q->where('status', true);
            }])->first();
            // dd($page_section);   
        return view('frontend.pages.about', compact(
            'page_title',
            'page_section'
        ));
    }

    /**
     * Method for view the manager
     * @return view
     */
    public function manager()
    {
        $page_title           = "Manager";
        return view('frontend.pages.manager', compact(
            'page_title',
        ));
    }

    /**
     * Method for view the about page
     * @return view
     */
    public function services()
    {
        
        $page_title             = "Services";
         $page_section       = SetupPage::where('slug', 'services')->with(['sections' => function ($q) {
                $q->where('status', true);
            }])->first();
        return view('frontend.pages.services-page', compact(
            'page_title',
            'page_section'
        ));
    }

    /**
     * Method for view the contact page
     * @return view
     */
    public function contact()
    {
        $page_title             = "Contact";
         $page_section       = SetupPage::where('slug', 'contact')->with(['sections' => function ($q) {
                $q->where('status', true);
            }])->first();
       
        return view('frontend.pages.contact-page', compact(
            'page_title',
            'page_section'
        ));
    }

    /**
     * Method for view the about page
     * @return view
     */
    public function blog()
    {
        $page_title             = "Blogs";
        $blogs                  = Blog::where('status', true)->paginate(6);
         $page_section       = SetupPage::where('slug', 'blog')->with(['sections' => function ($q) {
                $q->where('status', true);
            }])->first();
        return view('frontend.pages.blog-page', compact(
            'page_title',
            'blogs',
            'page_section'
        ));
    }

    /**
     * Method for show the blog details page
     * @param string $slug
     * @param \Illuminate\Http\Request $request
     */
    public function blogDetails($slug)
    {
        $page_title             = "Blog Details";
        $blog                   = Blog::where('slug', $slug)->first();
        if (!$blog) abort(404);
        $category               = BlogCategory::withCount('blog')->where('status', true)->get();
        $recent_posts           = Blog::where('status', true)->where('slug', '!=', $slug)->get();
        return view('frontend.pages.blog-details', compact(
            'page_title',
            'blog',
            'category',
            'recent_posts',
        ));
    }
    /**
     * Method for get the blogs using category
     * @param string $slug
     * @param \Illuminate\Http\Request $request
     */
    public function blogCategory($slug)
    {
        $page_title         = "Blog Category";
        $blog_category      = BlogCategory::where('slug', $slug)->first();

        if (!$blog_category) abort(404);
        $blogs              = Blog::where('category_id', $blog_category->id)->get();

        return view('frontend.pages.blog-category', compact(
            'page_title',
            'blog_category',
            'blogs',
        ));
    }

    /**
     * Method for contact request
     * @param string
     * @param \Illuminate\Http\Request $request
     */
    public function contactRequest(Request $request)
    {

        $validator        = Validator::make($request->all(), [
            'name'        => "required|string|max:255|unique:contact_requests",
            'email'       => "required|string|email|max:255|unique:contact_requests",
            'message'     => "required|string|max:5000",
        ]);
        if ($validator->fails()) return back()->withErrors($validator)->withInput();
        $validated = $validator->validate();
        try {
            ContactRequest::create([
                'name'            => $validated['name'],
                'email'           => $validated['email'],
                'message'         => $validated['message'],
                'created_at'      => now(),
            ]);
        } catch (Exception $e) {
            return back()->with(['error' => ['Failed to Contact Request. Try again']]);
        }
        return back()->with(['success' => ['Contact Request successfully send!']]);
    }

    /**
     * Method for show useful links
     */
    public function link($slug)
    {
        $link       = UsefulLink::where('slug', $slug)->first();
        if (!$link) return back()->with(['error' => ['Link not found.']]);
        $page_title = ucwords(strtolower(str_replace("_", " ", $link->type)));

        return view('frontend.pages.link', compact(
            'link',
            'page_title'
        ));
    }

    public function subscribe(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'email'     => "required|string|email|max:255|unique:subscribes",
        ]);

        if ($validator->fails()) return redirect()->back()->withErrors($validator)->withInput();

        $validated = $validator->validate();
        try {
            Subscribe::create([
                'email'         => $validated['email'],
                'created_at'    => now(),
            ]);
        } catch (Exception $e) {
            return redirect()->back()->with(['error' => ['Failed to subscribe. Try again']]);
        }

        return redirect()->back()->with(['success' => ['Subscription successful!']]);
    }

    /**
     * Method for search doctor
     * @param string $slug
     * @param \Illuminate\Http\Request  $request
     */
    public function searchParlour(Request $request)
    {
      

        $page_title             = "Find Salon";
        $areas                  = Area::where('status', true)->get();
        $message                = Session::get('message');

        $validator = Validator::make($request->all(), [
            'area'          => 'nullable',
            'name'          => 'nullable',
        ]);
        if ($validator->fails()) {
            return back()->with(['error' => ['Something went wrong! Please try again.']]);
        }
        if ($request->area && $request->name) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->where('name', 'like', '%' . $request->name . '%')->get();
        } else if ($request->area) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->get();
        } else {
            $parlour_lists    = ParlourList::where('name', 'like', '%' . $request->name . '%')->get();
        }
        $areaString     = $request->area;
        $nameString     = $request->name;

        return view('frontend.pages.find-parlour', compact(
            'page_title',
            'areas',
            'areaString',
            'nameString',
            'parlour_lists',
            'message'
        ));
    }

    public function changeLanguage($lang = null)
    {
        $language = Language::where('code', $lang)->first();
        session()->put('local', $lang);
        return redirect()->back();
    }

    public function contactMessageSend(Request $request)
    {

        $validated = Validator::make($request->all(), [
            'name'      => "required|string|max:255",
            'email'     => "required|email|string|max:255",
            'message'   => "required|string|max:5000",
        ])->validate();

        try {
            ContactRequest::create($validated);
        } catch (Exception $e) {
            return back()->with(['error' => ['Failed to send message. Please Try again']]);
        }

        return back()->with(['success' => ['Message send successfully!']]);
    }

    public function languageSwitch(Request $request)
    {
        $code = $request->target;
        $language = Language::where("code", $code)->first();
        if (!$language) {
            return back()->with(['error' => ['Oops! Language Not Found!']]);
        }
        Session::put('local', $code);
        Session::put('local_dir', $language->dir);

        return back()->with(['success' => ['Language Switch to ' . $language->name]]);
    }
}
