<?php

namespace App\Http\Controllers\Api\V1\Vendor;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Http\Helpers\Api\Helpers as ApiResponse;
use App\Models\Vendor\Manager;
use Illuminate\Support\Facades\Notification;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use App\Notifications\ManagerNotification;
use Exception;
use Illuminate\Foundation\Auth\RegistersUsers;

class ManagerController extends Controller
{

    use RegistersUsers;

    public function index()
    {

        $managers = Manager::auth()->with('parlourlist')->orderByDesc('id')->get();

        $manager_image_path   = [
            'base_url'      => url('/'),
            'path_location' => files_asset_path_basename('manager-profile'),
            'default_image' => files_asset_path_basename('default')
        ];
        $data = [
            'manager-image-path'    => $manager_image_path,
            'data'                   => $managers
        ];
        $message =  ['success' => [__('Manager List')]];
        return ApiResponse::success($message, $data);
    }
    /**
     * Method to register a new manager (API Compatible)
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function store(Request $request)
    {
        // Validate the incoming request
        $validator = Validator::make($request->all(), [
            'firstname' => 'required|string|max:60',
            'lastname'  => 'required|string|max:60',
            'password'  => 'required|string|min:6',
            'email'     => 'required|string|email|max:150|unique:users,email',
            'image'      => "nullable|image|mimes:jpg,png,jpeg,webp|max:10240",
        ]);



        if ($validator->fails()) {
            $error = ['error' => [$validator->errors()->all()]];
            return ApiResponse::onlyValidation($error);
        }

        // Prepare validated data
        $validated = $validator->validated();
        $validated['password'] = Hash::make($validated['password']);
        $validated['username'] = make_username($validated['firstname'], $validated['lastname'], "managers");
        $validated['vendor_id'] = $request->vendor_id;

        if ($request->hasFile('image')) {
            $image = upload_file($validated['image'], 'manager-profile');
            $upload_image = upload_files_from_path_dynamic([$image['dev_path']], 'manager-profile');
            delete_file($image['dev_path']);
            $validated['image']     = $upload_image;
        }

        try {
            // Create the manager
            $manager = Manager::create($validated);

            try {
                // Notify the manager via email
                Notification::route('mail', $validated['email'])->notify(new ManagerNotification($validated, $request->password));
            } catch (Exception $e) {
            }

            $data = [
                'manager' => $manager,
            ];
            $message    =  ['success' => [__('Manager registered successfully')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {
            $message =  ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
    }

    /**
     * Method for deleting the Manager information (API).
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function delete($slug)
    {
        try {

            $manager = Manager::find($slug);

            if (!$manager) {
                $message = ['error' => [__('Manager not found')]];
                return ApiResponse::onlyError($message);
            }


            $manager->delete();


            $data = ['Manager' => $manager];
            $message = ['success' => [__('Manager deleted successfully')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {

            $message = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
    }


    /**
     * Profile Update
     * @method POST
     * @param Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        $manager = Manager::find($id);
        if (!$manager) {
            $message = ['error' => [__('Manager not found')]];
            return ApiResponse::onlyError($message);
        }
        $validator = Validator::make($request->all(), [
            'firstname' => 'required|string|max:60',
            'lastname'  => 'required|string|max:60',
        ]);


        if ($validator->fails()) {
            $error = ['error' => [$validator->errors()->all()]];
            return ApiResponse::onlyValidation($error);
        }

        $validated = $validator->validated();

        $validated['firstname']   = $validated['firstname'];
        $validated['lastname']    = $validated['lastname'];

        try {
            $manager->update($validated);
        } catch (\Throwable $th) {
            $error = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::error($error);
        }

        $message =  ['success' => [__('Manager successfully updated')]];
        return ApiResponse::onlySuccess($message);
    }
}
