<?php

namespace App\Http\Controllers\Api\V1\Vendor\Auth;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;

use Exception;
use App\Constants\GlobalConst;
use Illuminate\Support\Carbon;
use App\Models\UserPasswordReset;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Validator;
use Illuminate\Validation\Rules\Password;
use App\Providers\Admin\BasicSettingsProvider;
use App\Http\Helpers\Api\Helpers as ApiResponse;
use App\Models\Vendor\Vendor;
use App\Models\Vendor\VendorPasswordReset;
use App\Notifications\User\Auth\PasswordResetEmail;

class ForgotPasswordController extends Controller
{
    /**
     * Send Verification code to user email/phone.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function sendCode(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'credentials'   => "required|string|max:100",
        ]);

        if ($validator->fails()) {
            $message =  ['error' => $validator->errors()->all()];
            return ApiResponse::onlyValidation($message);
        }

        $column = "mobile";
        if(check_email($request->credentials)) $column = "email";
        $user = Vendor::where($column,$request->credentials)->first();

        if (!$user) {
            $message =  ['error' => ["Vendor doesn't exists"]];
            return ApiResponse::onlyError($message);
        }

        $token = generate_unique_string("vendor_password_resets", "token", 80);
        $code = generate_random_code();

        try {
            VendorPasswordReset::where("vendor_id",$user->id)->delete();

            $password_reset = VendorPasswordReset::create([
                'vendor_id'       => $user->id,
                'token'         => $token,
                'code'          => $code,
            ]);

            $user->notify(new PasswordResetEmail($user, $password_reset));

        } catch (Exception $e) {
            info($e);
            $message = ['error' =>  [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
        $data = ['user' => $password_reset];
        $message = ['success' =>  [__('Verification otp code sended to your email')]];

        return ApiResponse::success($message, $data);
    }




    public function resendCode(Request $request)
    {
        $validator = Validator::make($request->all(),[
            'token'     => "required|string|exists:vendor_password_resets,token"
        ]);

        if($validator->fails()) {
            return ApiResponse::error($validator->errors()->all(),[]);
        }
        $validated = $validator->validate();
        $password_reset = VendorPasswordReset::where('token',$validated['token'])->first();

        if(!$password_reset) return ApiResponse::error(['Request token is invalid'],[],400);

        if(Carbon::now() <= $password_reset->created_at->addMinutes(GlobalConst::USER_PASS_RESEND_TIME_MINUTE)) {
            return ApiResponse::error([__('You can resend verification code after').Carbon::now()->diffInSeconds($password_reset->created_at->addMinutes(GlobalConst::USER_PASS_RESEND_TIME_MINUTE)). __('seconds')],[],400);
        }

        DB::beginTransaction();
        try{
            $update_data = [
                'code'          => generate_random_code(),
                'created_at'    => now(),
                'token'         => $validated['token'],
            ];
            DB::table('vendor_password_resets')->where('token',$validated['token'])->update($update_data);
            $password_reset->vendor->notify(new PasswordResetEmail($password_reset->vendor,(object) $update_data));
            DB::commit();
        }catch(Exception $e) {
            DB::rollback();
            return ApiResponse::error([__('Something went wrong! Please try again')],[],500);
        }

        return ApiResponse::success([__('OTP resend success')],['token' => $validated['token']],200);
    }


    /**
     * OTP Verification.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function verifyCode(Request $request)
    {
        $token = $request->token;
        $request->merge(['token' => $token]);
        $rules = [
            'token'         => "required|string|exists:vendor_password_resets,token",
            'otp'          => "required|numeric|exists:vendor_password_resets,code",
        ];

        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $message =  ['error' => $validator->errors()->all()];
            return ApiResponse::onlyValidation($message);
        }

        $basic_settings = BasicSettingsProvider::get();
        $otp_exp_seconds = $basic_settings->otp_exp_seconds ?? 0;

        $password_reset = VendorPasswordReset::where("token", $token)->first();

        if (Carbon::now() >= $password_reset->created_at->addSeconds($otp_exp_seconds)) {
            foreach (VendorPasswordReset::get() as $item) {
                if (Carbon::now() >= $item->created_at->addSeconds($otp_exp_seconds)) {
                    $item->delete();
                }
            }
            $message = ['error' => [__('Session expired. Please try again')]];
            return ApiResponse::onlyError($message);
        }
        if ($password_reset->code != $request->otp) {
            $message = ['error' => [__('Verification OTP invalid')]];
            return ApiResponse::onlyError($message);
        }
        $data = ['password_reset_data' => $password_reset];
        $message = ['success' => [__('OTP verification successful')]];
        return ApiResponse::success($message, $data);
    }

    /**
     * Password Reset.
     *
     * @method POST
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function resetPassword(Request $request)
    {
        if ($request->password != $request->password_confirmation) {
            $message = ['error' => [__('Oops password does not match')]];
            return ApiResponse::onlyError($message);
        }

        $token = $request->token;
        $basic_settings = BasicSettingsProvider::get();
        $password_rule = "required|string|min:6|confirmed";

        if ($basic_settings->secure_password) {
            $password_rule = ["required", Password::min(8)->letters()->mixedCase()->numbers()->symbols()->uncompromised(), "confirmed"];
        }
        $request->merge(['token' => $token]);
        $rules = [
            'token'         => "required|string|exists:vendor_password_resets,token",
            'password'      => $password_rule,
        ];
        $validator = Validator::make($request->all(), $rules);

        if ($validator->fails()) {
            $message =  ['error' => $validator->errors()->all()];
            return ApiResponse::onlyError($message);
        }


        $password_reset = VendorPasswordReset::where("token", $token)->first();
        if (!$password_reset) {
            $message = ['error' => [__('Invalid Request. Please try again')]];
            return ApiResponse::onlyError($message);
        }
        try {
            $password_reset->vendor->update(['password' => Hash::make($request->password)]);
            $password_reset->delete();
        } catch (Exception $e) {
            info($e);
            $message = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
        $message = ['success' => [__('Password reset success. Please login with new password')]];
        return ApiResponse::onlySuccess($message);
    }
}
