<?php

namespace App\Http\Controllers\Api\V1\User;

use Exception;
use App\Constants\GlobalConst;
use App\Http\Helpers\Response;
use App\Models\Admin\Language;
use App\Models\Admin\AppSettings;
use App\Http\Controllers\Controller;
use Illuminate\Support\Facades\Route;
use App\Models\Admin\AppOnboardScreens;
use App\Providers\Admin\CurrencyProvider;
use App\Providers\Admin\BasicSettingsProvider;
use App\Models\Admin\Area;
use App\Models\Admin\ParlourList;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use App\Models\ParlourBooking;
use Illuminate\Support\Facades\DB;
use App\Models\Admin\ParlourHasService;
use App\Models\Admin\ParlourListHasSchedule;
use App\Models\Admin\UsefulLink;

class SettingController extends Controller
{
    protected $languages;
    public function __construct()
    {
        $this->languages = Language::get();
    }
    public function basicSettings()
    {
        $basic_settings = BasicSettingsProvider::get()->only(['id','min_stuff', 'site_name', 'base_color', 'secondary_color','vendor_base_color','vendor_secondary_color','manager_base_color','manager_secondary_color', 'site_title', 'timezone', 'site_logo','vendor_site_logo','vendor_site_logo_dark','manager_site_logo','manager_site_logo_dark', 'site_logo_dark', 'site_fav',  'manager_site_fav', 'vendor_site_fav','site_fav_dark','manager_site_fav_dark','vendor_site_fav_dark','vendor_registration','vendor_agree_policy','user_registration','agree_policy']);

        $languages = Language::select(['id', 'name', 'code', 'status'])->get();

        $app_settings = AppSettings::first();
        $onboard_screens = AppOnboardScreens::select("title", "image", "status")->where("status", GlobalConst::ACTIVE)->orderByDesc('id')->get();
        $onboard_screens->makeHidden(['editData']);

        $languages = $this->languages;
        $lang = request()->lang;
        $default = 'en';
        $onboard_screens_data = [];
        foreach ($onboard_screens as $value) {
            $onboard_screens_data[] = [
                'title' => $value->title->language->$lang->title  ?? '',
                'image' => $value->image  ?? '',
                'status' => $value->status  ?? '',
            ];
        }

        $base_cur = CurrencyProvider::default()->first();
        $base_cur->makeHidden(['admin_id', 'country', 'name', 'created_at', 'updated_at', 'type', 'flag', 'sender', 'receiver', 'default', 'status', 'editData']);

        $app_image_paths = [
            'base_url'          => url("/"),
            'path_location'     => files_asset_path_basename("app-images"),
            'default_image'     => files_asset_path_basename("default"),
        ];


        return Response::success([__("Basic settings fetch successfully!")], [
            'basic_settings'    => $basic_settings,
            'base_cur'          => $base_cur,
            'web_links'         => [
                'about-us'          => Route::has('frontend.about') ? route('frontend.about') : url('/'),
                'contact-us'        => Route::has('frontend.contact') ? route('frontend.contact') : url('/'),
                'contact-us'        => Route::has('frontend.contact') ? route('frontend.contact') : url('/'),
                'privacy-policy'    => setRoute('frontend.link',UsefulLink::where('type',GlobalConst::USEFUL_LINK_PRIVACY_POLICY)->first()?->slug),
            ],
            'languages'         => $languages,
            'splash_screen'     => $app_settings,
            'onboard_screens'   => $onboard_screens_data,
            'image_paths'       => [
                'base_path'         => url("/"),
                'path_location'     => files_asset_path_basename("image-assets"),
                'default_image'     => files_asset_path_basename("default"),
            ],
            'app_image_paths'   => $app_image_paths,
        ], 200);
    }

    public function splashScreen()
    {
        $app_settings = AppSettings::select('splash_screen_image as image', 'version')->first();

        $image_paths = [
            'base_url'          => url("/"),
            'path_location'     => files_asset_path_basename("app-images"),
            'default_image'     => files_asset_path_basename("default"),
        ];

        return Response::success([__('Splash screen data fetch successfully!')], [
            'splash_screen' => $app_settings,
            'image_paths'   => $image_paths,
        ], 200);
    }

    public function onboardScreens()
    {
        $onboard_screens = AppOnboardScreens::select("title", "sub_title", "image", "status")->where("status", GlobalConst::ACTIVE)->orderByDesc('id')->get();
        $onboard_screens->makeHidden(['editData']);

        $image_paths = [
            'base_url'          => url("/"),
            'path_location'     => files_asset_path_basename("app-images"),
            'default_image'     => files_asset_path_basename("default"),
        ];

        return Response::success([__('Onboard screen data fetch successfully!')], [
            'onboard_screens'   => $onboard_screens,
            'image_paths'       => $image_paths,
        ], 200);
    }

    public function getLanguages()
    {
        try {
            $api_languages = get_api_languages();
        } catch (Exception $e) {
            return Response::error([$e->getMessage()], [], 500);
        }

        return Response::success([__("Language data fetch successfully!")], [
            'languages' => $api_languages,
        ], 200);
    }

    /**
     * Method for get the parlour List
     */
    public function parlourList()
    {
        $areas          = Area::where('status', true)->orderBy("id")->get()->map(function ($data) {
            return [
                'id'        => $data->id,
                'slug'      => $data->slug,
                'name'      => $data->name,
                'status'    => $data->status,
            ];
        });

        $parlour_list   = ParlourList::where('status', true)->with('parlourImage','parlourBooking')->orderBy("id")->get()->map(function ($data) {
            return [
                'id'                => $data->id,
                'area_id'           => $data->area_id,
                'vendor_id'         => $data->vendor_id,
                'slug'              => $data->slug,
                'name'              => $data->name,
                'manager_name'      => $data->manager_name,
                'experience'        => $data->experience,
                'speciality'        => $data->speciality,
                'contact'           => $data->contact,
                'address'           => $data->address,
                'off_days'          => $data->off_days,
                'number_of_dates'   => $data->number_of_dates,
                'image'             => $data->image,
                'optional_image'    => $data->parlourImage->map(function ($image) {
                    return [
                        'id'        => $image->id,
                        'parlour_lists_id' => $image->parlour_lists_id,
                        'image'     => $image->image_path,
                    ];
                }),
                'parlour_rating'    => $data->parlourBooking->map(function ($rating) {
                    return [
                        'id'            => $rating->id, 
                        'review'        => $rating->review,
                   
                    ];
                }),
                'status'            => $data->status,
                'created_at'        => $data->created_at
            ];
        });
        $parlour_image_path   = [
            'base_url'      => url('/'),
            'path_location' => files_asset_path_basename('site-section'),
            'default_image' => files_asset_path_basename('default')
        ];
        return Response::success(['Data Fetch Successfuly.'], [
            'area'                      => $areas,
            'parlour_list'              => $parlour_list,
            'parlour_image_path'        => $parlour_image_path,
        ], 200);
    }

    /**
     * Method for search parlour
     */
    public function searchParlour(Request $request)
    {

        $validator      = Validator::make($request->all(), [
            'area'          => 'nullable',
            'name'          => 'nullable',
        ]);
        if ($validator->fails()) {
            return Response::error($validator->errors()->all(), []);
        }
        if ($request->area && $request->name) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->where('name', 'like', '%' . $request->name . '%')->get();
        } else if ($request->area) {
            $parlour_lists    = ParlourList::where('area_id', $request->area)->get();
        } else {
            $parlour_lists    = ParlourList::where('name', 'like', '%' . $request->name . '%')->get();
        }
        if ($parlour_lists->isEmpty()) {
            return Response::error(['Salon not found!'], [], 404);
        }
        return Response::success(['Salon Find Successfully!'], $parlour_lists, 200);
    }
    /**
     * Method for check the schedule data
     */
    public function checkSchedule(Request $request)
    {
        $validator          = Validator::make($request->all(), [
            'date'  => 'required'
        ]);
        if ($validator->fails()) return Response::error($validator->errors()->all());
        $validated          = $validator->validate();
        $booking_data       = ParlourBooking::select('schedule_id', DB::raw('count(*) as total'))
            ->where('date', $validated['date'])
            ->groupBy('schedule_id')
            ->get();
        return Response::success(['Booking data fetch successfully.'], ['data' => $booking_data], 200);
    }

    /**
     * Method for parlour Service and schedule
     */
    public function scheduleService($id)
    {

        $parlour_has_service   = ParlourHasService::where('parlour_list_id', $id)->orderBy("id")->get()->map(function ($data) {
            return [
                'id'                => $data->id,
                'parlour_list_id'   => $data->parlour_list_id,
                'service_name'      => $data->service_name,
                'price'             => $data->price,
            ];
        });

        $parlour_has_schedule   = ParlourListHasSchedule::where('parlour_list_id', $id)->where('status', true)->orderBy("id")->get()->map(function ($data) {
            return [
                'id'                => $data->id,
                'parlour_list_id'   => $data->parlour_list_id,
                'from_time'         => $data->from_time,
                'to_time'           => $data->to_time,
                'max_client'        => $data->max_client,
                'status'            => $data->status
            ];
        });

        return Response::success(['Data Fetch Successfuly.'], [
            'parlour_has_service'       => $parlour_has_service,
            'parlour_has_schedule'      => $parlour_has_schedule,
        ], 200);
    }
}
