<?php

namespace App\Http\Controllers\Api\V1\Manager;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use App\Models\ParlourBooking;
use App\Models\Admin\ParlourList;
use App\Notifications\EmailNotification;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Notification;
use App\Models\Admin\BasicSettings;
use App\Models\Vendor\ParlourHasStuff;
use Exception;
use App\Http\Helpers\Api\Helpers as ApiResponse;
use App\Models\User;
use App\Constants\GlobalConst;

class BookingRequestController extends Controller
{
    public function index(Request $request)
    {
        try {
            $vendor_id = auth()->user()->vendor_id;
            $parlour_id = ParlourList::where('manager_id', auth()->id())->value('id');

            $booking_data = ParlourBooking::with('parlour', 'schedule', 'user')
                ->where('vendor_id', $vendor_id)
                ->where('parlour_id', $parlour_id)
                ->orderByDesc('id')
                ->get();

            $parlour_has_stuff = ParlourHasStuff::where('vendor_id', $vendor_id)
                ->where('parlour_list_id', $parlour_id)
                ->get();



            $image_paths = [
                'base_url'          => url("/"),
                'path_location'     => files_asset_path_basename("user-profile"),
                'default_image'     => files_asset_path_basename("profile-default"),
            ];

            $data = [
                'path'          => $image_paths,
                'booking_data' => $booking_data,
                'parlour_has_stuff' => $parlour_has_stuff
            ];

            $message = ['success' => [__('Manager Booking Request')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {

            $message = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
    }

    public function statusUpdate(Request $request)
    {

        $validator = Validator::make($request->all(), [
            'status' => 'required|integer',
            'trx_id' => 'required',
        ]);

        if ($validator->fails()) {
            $error = ['error' => [$validator->errors()->all()]];
            return ApiResponse::onlyValidation($error);
        }

        try {
            $data = ParlourBooking::with(['parlour', 'schedule', 'payment_gateway'])
                ->where('trx_id', $request->trx_id)
                ->first();

            if (!$data) {
                $message = ['error' => [__('Data Not Found!')]];
                return ApiResponse::onlyError($message);
            }

            $validated = $validator->validate();
            $basic_setting = BasicSettings::first();

            $data->update([
                'status' => $validated['status'],
            ]);

            $user = User::where('id', $data->user_id)->first();
            if ($basic_setting->email_notification == true) {
                try {

                    Notification::route("mail", $user->email)->notify(new EmailNotification($user, $data, $data->trx_id));
                } catch (Exception $e) {
                }
            }

            $message = ['success' => [__('Booking Status Updated Successfully.')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {
            $message = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
    }

    public function assignTempStuff(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'parlour_booking_id' => 'required|integer',
            'stuff_id.*' => 'required|integer',
        ]);

        if ($validator->fails()) {
            $error = ['error' => [$validator->errors()->all()]];
            return ApiResponse::onlyValidation($error);
        }

        $validated = $validator->validate();
        $parlour_booking_id = $validated['parlour_booking_id'];

        $previous_stuff = ParlourBooking::where('stuff_id', $request->stuff_id)->first();


        try {

            if ($previous_stuff) {

                ParlourBooking::where('stuff_id', $request->stuff_id)->update([
                    'stuff_id'            => null,
                    'stuff_assign_status' => 0,
                ]);
            }


            // Assign new stuff
            ParlourBooking::where('id', $parlour_booking_id)->update([
                'stuff_id'            => $request->stuff_id,
                'stuff_assign_status' => GlobalConst::STUFFASSIGN,
            ]);


            $message = ['success' => [__('Stuff Assigned Successfully!')]];
            return ApiResponse::success($message);
        } catch (Exception $e) {
            $message = ['error' => [__('Something went wrong. Please try again.')]];
            return ApiResponse::onlyError($message);
        }
    }

    public function myParlour()
    {
        try {
            $image_paths = [
                'base_url'      => url("/"),
                'path_location' => files_asset_path_basename("site-section"),
                'default_image' => files_asset_path_basename("default"),
            ];

            $manager_id = auth()->user()->id;
            $data = ParlourList::with(['schedules', 'stuff', 'services', 'area', 'vendor', 'manager'])
                ->where('manager_id', $manager_id)
                ->get(); // Use get() instead of first()

            if ($data->isEmpty()) { // Check if collection is empty
                $message = ['success' => [__('No data found for the manager.')]];
                return ApiResponse::success($message, []);
            }

            // Append image paths to each item in the collection
            $data = $data->map(function ($item) use ($image_paths) {
                $item['image_paths'] = $image_paths;
                return $item;
            });

            $message = ['success' => [__('Salon details retrieved successfully.')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {
            $message = ['error' => [__('Something went wrong. Please try again.')]];
            return ApiResponse::onlyError($message);
        }
    }


    public function stuff()
    {
        try {
            // Check if the user is authenticated
            $user = auth()->user();
            if (!$user) {
                $message = ['error' => [__('User is not authenticated.')]];
                return ApiResponse::onlyError($message);
            }

            // Get vendor ID and parlour ID
            $vendor_id = $user->vendor_id;
            $parlour_id = ParlourList::where('manager_id', $user->id)->value('id');

            if (!$parlour_id) {
                $message = ['error' => [__('No Salon found for the current manager.')]];
                return ApiResponse::onlyError($message);
            }

            // Retrieve stuff data
            $stuff = ParlourHasStuff::with(['parlour', 'parlour_booking'])
                ->where('vendor_id', $vendor_id)
                ->where('parlour_list_id', $parlour_id)
                ->get();

            // Check if stuff is empty
            if ($stuff->isEmpty()) {
                $message = ['error' => [__('No stuff found for the parlour.')]];
                return ApiResponse::success($message, []);
            }

            // Image paths
            $image_paths = [
                'base_url'      => url("/"),
                'path_location' => files_asset_path_basename("stuff-profile"),
                'default_image' => files_asset_path_basename("profile-default"),
            ];

            // Transform data
            $filteredData = $stuff->map(function ($item) use ($image_paths) { // Pass $image_paths using `use`
                return [
                    'stuff_name'   => $item->stuff_name,
                    'parlour_name' => optional($item->parlour)->name,
                    'assign_status' => $item->assign_status,
                    'stuff_image'  => $item->image,
                    'image_path'   => $image_paths,
                ];
            });

            $message = ['success' => [__('Stuff data retrieved successfully.')]];
            return ApiResponse::success($message, $filteredData);
        } catch (Exception $e) {


            $message = ['error' => [__('Something went wrong. Please try again.')]];
            return ApiResponse::onlyError($message);
        }
    }

    public function remarkAdd(Request $request)
    {
        $parlour = ParlourBooking::find($request->id);

        try {
            $parlour->update(
                [
                    'notes' => $request->remarks,
                ]
            );

        
        } catch (Exception $e) {
        }
        $message = ['success' => [__('Remarks added to this booking request.')]];
        return ApiResponse::success($message, []);

    }




    public function history(Request $request)
    {
        try {
            $vendor_id = auth()->user()->id;
            $booking_data = ParlourBooking::where('vendor_id', $vendor_id)
                ->orderByDesc('id')
                ->get();

            $data = ['booking_data' => $booking_data];

            $message = ['success' => [__('Booking Data')]];
            return ApiResponse::success($message, $data);
        } catch (Exception $e) {
            $message = ['error' => [__('Something went wrong! Please try again')]];
            return ApiResponse::onlyError($message);
        }
    }
}
